/* RoombaLib Nov 2012 - Ralf Stoffels
 * 
 * The library contains basic control API for Roomba Vaccum Cleaner robots
 * controlled by an Arduino-Board.
 * It is based on the the Arduino example in the book "Hacking Roomba"
 * In order to reserve the serial interface for communication with the PC
 * this library uses regular IO Pins. This function is provided by the 
 * library "NewSoftSerial" of Mikal Hart (http://arduiniana.org/libraries/newsoftserial/)
 
 * How to instatiate an object in the constructor of an object is described here: 
 * http://arduino.cc/forum/index.php/topic,41519.0.html 
*/

#include <roombaRS.h>
#include "WProgram.h"

roombaRS::roombaRS() : sciSerial(2,3) {  // this is needed to instantiate the sciSerial Object in the constructor
  ledPin = 13;  // define which pin will have an LED connected to it for simple status
}

void roombaRS::wakeUp()
{
  // define "sciSerial" as a NewSoftSerial serial port on those pins
                                            
  int ddPin = 4;

  pinMode(ledPin, OUTPUT);    // set the mode of the pin "ledPin"
  pinMode(ddPin,  OUTPUT);    // set the mode of the pin "ddPin"
  sciSerial.begin(57600);    // set the baud rate of the port "sciSerial"
                              // (defined above)
  
  digitalWrite(ledPin, HIGH); // turn on the LED to say we're alive
  
  // wake up the robot by flashing the "data detect" pin
  digitalWrite(ddPin, HIGH);
  delay(100);
  digitalWrite(ddPin, LOW);
  delay(500);
  digitalWrite(ddPin, HIGH);
  delay(2000);
  
  // set up the Roomba OI to receive commands (see OI manual)
  sciSerial.print(128, BYTE);  // START (always first command)
  delay(50);
  sciSerial.print(130, BYTE);  // CONTROL (same as "Safe Mode")
  delay(50);
  
  // turn off the LED to confirm that we're done setting up Roomba
  digitalWrite(ledPin, LOW);
  delay(100);
}

void roombaRS::Sleep() {
  sciSerial.print(133, BYTE);  // POWER (back to passive Mode)
}  

void roombaRS::findCharger() {
  sciSerial.print(143,BYTE); // Roomba tries to find its dock
}

void roombaRS::brushesON() {
  sciSerial.print(138, BYTE); // Roomba's MOTOR
  sciSerial.print(255,BYTE); // Velocity high byte, HEX 00
}

void roombaRS::brushesOFF() {
  sciSerial.print(138, BYTE); // Roomba's MOTOR command
  sciSerial.print(0,BYTE); // Velocity high byte, HEX 00
}

void roombaRS::goDistance(int mm) {   // Distance in mm, >0 foreward move, <0 backward move
  int drivetime;
  drivetime = abs(5*mm);      //time in ms 
  if (mm > 0) goForward();
  else goBackward();
  delay(drivetime);
  stopMoving();
}

void roombaRS::turnLeft(int degrees) {
  float ftime;
  int drivetime;
  ftime = (float)degrees/360*4000;      //time in ms 
  drivetime = (int)ftime;
  spinLeft();
  delay(drivetime);
  stopMoving();
}

void roombaRS::turnRight(int degrees) {
  float ftime;
  int drivetime;
  ftime = (float)degrees/360*4000;  //time in ms 
  drivetime = (int)ftime;
  spinRight();
  delay(drivetime);
  stopMoving();
}
   
void roombaRS::goForward() {
  sciSerial.print(137, BYTE); // Roomba's DRIVE command
  sciSerial.print(0x00,BYTE); // Velocity high byte, HEX 00
  sciSerial.print(0xc8,BYTE); // Velocity low byte,  HEX C8 (00C8=200mm/sec)
  sciSerial.print(0x80,BYTE); // Radius high byte,   HEX 80
  sciSerial.print(0x00,BYTE); // Radius low byte,    HEX 00 (8000=straight)
}

void roombaRS::goBackward() {
  sciSerial.print(137, BYTE); // Roomba's DRIVE command
  sciSerial.print(0xff,BYTE);
  sciSerial.print(0x38,BYTE); // Velocity FF38 = -200 mm/sec (backwards)
  sciSerial.print(0x80,BYTE);
  sciSerial.print(0x00,BYTE); // Radius 8000 = go straight
}

void roombaRS::spinLeft() {
  sciSerial.print(137, BYTE); // Roomba's DRIVE command
  sciSerial.print(0x00,BYTE);   
  sciSerial.print(0xc8,BYTE); // Velocity of spin 00C8 = 200 mm/sec
  sciSerial.print(0x00,BYTE);
  sciSerial.print(0x01,BYTE); // Radius 0001 = spin left
}

void roombaRS::spinRight() {
  sciSerial.print(137, BYTE); // Roomba's DRIVE command
  sciSerial.print(0x00,BYTE);
  sciSerial.print(0xc8,BYTE); // Velocity of spin 00C8 = 200 mm/sec
  sciSerial.print(0xff,BYTE);
  sciSerial.print(0xff,BYTE); // Radius FFFF = spin right
}

void roombaRS::stopMoving() {
  sciSerial.print(137, BYTE); // Roomba's DRIVE command
  sciSerial.print(0x00,BYTE);
  sciSerial.print(0x00,BYTE); // Velocity 0000 = stopped
  sciSerial.print(0x00,BYTE);
  sciSerial.print(0x00,BYTE); // Radius 0000 = nothing
}

int roombaRS::Bump() {
  char sensorbytes[10];  // define a 10-byte array for holding sensor data

  sciSerial.print(142, BYTE); // Roomba's SENSORS command
  sciSerial.print(1,   BYTE); // get sensor packet group 1, which is 10
                              // bytes long. This packet group starts with
                              // packet 7, of which the first two bits are
                              // for the bumpleft and bumpright sensors

  delay(64);  // wait for sensor data
  
  // before storing the new sensor data, wipe the old sensor data
  char i = 0;                  // start at position 0 in the array
  while (i < 10) {             // do a loop until i=10
    sensorbytes[i++] = 0;      // set the byte to 0 and then increment
                               // the loop counter
  } // this bracket is the end of the "wipe data" loop
  
  // go back to position 0 to prepare to receive the new data
  i=0;
  
  // this is the loop to actually read the incoming data and store it
  while(sciSerial.available()) {  // start the loop - check if there is
                                  // data coming in,
    int c = sciSerial.read();     // read it and store the current
                                  // byte as "c"
    
       // error routine to flash the LED 5 times if c= -1
       // (this shouldn't happen)
       if( c==-1 ) {
         for( int i=0; i<5; i ++ ) {    // loop to flash LED 5 times
           digitalWrite(ledPin, HIGH);
           delay(50);
           digitalWrite(ledPin,  LOW);
           delay(50);
      } 
    } 
    
    // since we've read a byte, save it in the sensorbytes array at
    // position "i", and then increment "i"
    sensorbytes[i++] = c;
  } 
  
  if (sensorbytes[0] & 0x01) return RIGHT
  else
  if (sensorbytes[0] & 0x02) return LEFT
  else
  return 0;
}  

void roombaRS::note(char noteID, int octave, int noteLength) {
  // Note ID can be "A" ... "G" or "a" ... "g" for half notes
  // NoteLength specifies the fraction if a full note e.g. 16 means 1/16
  int duration;
  int noteCode;

  if (octave<1) octave=1;
  if (octave>7) octave=7; 
  
  noteCode=noteID+(octave-1)*12; // Note "C" in octave 1 has the code 36
  
  duration = 2048 / noteLength;
  
  if (noteID!=0) {
     sciSerial.print(140, BYTE); // Roomba's SONG command
     sciSerial.print(15,BYTE);
     sciSerial.print(1,BYTE);
     sciSerial.print(noteCode,BYTE);
     sciSerial.print(duration/16,BYTE);
     sciSerial.print(141, BYTE); // Roomba's PLAY command
     sciSerial.print(15, BYTE);
  } 
  delay(duration+50);
  
  Serial.println(noteCode);
}

void roombaRS::fanfare() {
  note(D,3,4);
  note(G,3,4);
  note(G,3,8);
  note(A,3,8);
  note(H,3,4);
  note(G,3,4);
  note(D,4,2);
  note(H,3,4);
}

//===================================================================
// Deutsche Programmierschnittstelle für Kinder und andere Einsteiger

void roombaRS::wachAuf() {
  wakeUp();
}

void roombaRS::fahrStrecke(int mm) {
  goDistance(mm);
}

void roombaRS::rechts(int degrees) {
  turnRight(degrees);
}

void roombaRS::links(int degrees) {
   turnLeft(degrees);
}

void roombaRS::vor() {
   goForward();
}

void roombaRS::zurueck() {
   goBackward();
}  

void roombaRS::rotiereLinks(){
   spinLeft();
}

void roombaRS::rotiereRechts(){
   spinRight();
}

void roombaRS::schlafEin() {
   Sleep();
}


int roombaRS::wand() {
   return Bump();
}

void roombaRS::suchLader() {
   findCharger();
}

void roombaRS::saugerEIN() {
   brushesON();
}

void roombaRS::saugerAUS() {
   brushesOFF();
}