//===========================================================================
//    Fledermaus-Sensor with LoRaWAN and TTN Integration
//    https://surasto.de CC4.0-BY-NC-SA 2022
//    
//    Board: Adafruit Feather M0+ LoRa
//    
//============================================================================

#define DEBUG 1

#define MINUTES_AWAKE 120              // Determines for how long the scanner stays awake

//======== Adafruit FFT Library for SAMD21xx Processors (ARM M0) =====
#include "Adafruit_ZeroFFT.h"

//=========== Library for ADC with DMA ===============================
#include <Adafruit_ZeroDMA.h>

//=============== Modified LMIC Library for Featehr M0+ ==============
#include <lmic.h>
#include <hal/hal.h>
#include <SPI.h>
#include <avr/dtostrf.h>

//============= Library for RTC ====================================
#include <Wire.h>
#include <DS1337.h>

//========== Global FFT Signal storage =============================
#define SAMPLE_BLOCK_LENGTH 2048
int16_t fft_signal_averaged[SAMPLE_BLOCK_LENGTH/2];   // we need only half of the size since the spectrum get folded
int16_t idx[SAMPLE_BLOCK_LENGTH/2];
uint8_t vBattByte;

int loopcount=0;

//==========================================================================
//  Here go the EUI, DEVEUI and Appkey of the Thethingsnetwork V3 
//==========================================================================

// This EUI must be in little-endian format, so least-significant-byte
// first. When copying an EUI from ttnctl output, this means to reverse
// the bytes. For TTN issued EUIs the last bytes should be 0xD5, 0xB3,
// 0x70.
static const u1_t PROGMEM APPEUI[8]={ 0xFF, 0xFF, 0x00, 0xD0, 0x7E, 0xD5, 0xB3, 0x70 };
void os_getArtEui (u1_t* buf) { memcpy_P(buf, APPEUI, 8);}

// This should also be in little endian format, see above.
static const u1_t PROGMEM DEVEUI[8]={0x9A, 0xF7, 0x04, 0xD0, 0x7E, 0xD5, 0xB3, 0x70};
void os_getDevEui (u1_t* buf) { memcpy_P(buf, DEVEUI, 8);}

// This key should be in big endian format (or, since it is not really a
// number but a block of memory, endianness does not really apply). In
// practice, a key taken from ttnctl can be copied as-is.
// The key shown here is the semtech default key.
static const u1_t PROGMEM APPKEY[16] = { 0x49, 0xB9, 0xCB, 0x08, 0x04, 0x73, 0xFD, 0xCA, 0x57, 0x55, 0x63, 0xA7, 0x41, 0x5E, 0x18, 0x53 };
void os_getDevKey (u1_t* buf) {  memcpy_P(buf, APPKEY, 16);}

//================= Global Variabls for LMIC =======================
uint8_t ttn_data[49];    // This is the array which gets send to TTN -- 12x4 byte for sorted FFT data + 1 byte for Vbatt
static osjob_t sendjob;
const unsigned TX_INTERVAL = 60;

//================= Zero DMA Object for fast ADC =====================
Adafruit_ZeroDMA ADC_DMA;


//==========================================================================
// Measure Voltage and convert into ony byte to be send with teh FFT array
// NOTE: Since the ADC measurement interferes with the DMA Measurement I can 
//       call this measurement only once after wake up. 
// TODO: Find out how this could be done in between the DMA Measurements
//==========================================================================
void getVBatt() {
  float vbat = analogRead(7);   // Pin 7 is the ananlog pin connected to the battery
  vbat *= 2;                    // Voltage is devided by 2 by resistors - compensated here
  vbat *= 3.3;                  // Multiplied by reference Voltage
  vbat /= 1024;                 // Devided by ADC max code    
  vbat *= 10;                   // 33 means 3.3V
  vBattByte = (uint8_t) vbat;

  Serial.print("VBatt = ");
  Serial.println(vBattByte);

}


//===========================================================================
// Arduino setup() starts the OTAA Autentification at the Thethingsnetwork
// and initializes the ADC and the DMA
//===========================================================================
void setup() {
    byte clockSet = 0;
    int i;
    char c;
    
    Serial.begin(115200);

    //=========== Check for 30s if the user wants to set the clock ===============
    while ((i++<60)&&(clockSet==0)) {
       Serial.print(millis()/1000);
       Serial.println(F(": Send an <s> for Clock and Alarm Setting"));
       while (Serial.available()) {
          c = Serial.read();
          if (c == 's') clockSet = 1; 
       }
      delay(500);
    }

    getVBatt();     // Read out Battery Voltage once (cannot do repeatedly because of interfernece with ADC DMA
    
    if (clockSet) setClockAlarm();
    Serial.println(F("-------- Starting -----------"));

    // LMIC init
    os_init();
    // Reset the MAC state. Session and pending data transfers will be discarded.
    LMIC_reset();
    LMIC_setClockError(MAX_CLOCK_ERROR * 1 / 100);

    // Start job (sending automatically starts OTAA too)
    do_send(&sendjob);

    adc_init();
    dma_init();
    ADC_DMA.startJob();

}



void loop() {
  os_runloop_once();
  dma_polling();
  delay(100);

  if (isTimeToSleep(MINUTES_AWAKE)) gotoSleep();   // Check if sleeptime arrived

  #if DEBUG>0
  if (loopcount++ > 600) {
    loopcount = 0;
    sort_bins(fft_signal_averaged, idx);   // Sort largest bins to the top
    
    create_ttn_data(ttn_data, fft_signal_averaged, idx);   // Creates the ttn data array

    print_decoded_ttn_data();

    delay(1000);
    clear_fft_average();
  }
  #endif

}

#if DEBUG>0
//============ for debug purpose ================
void print_decoded_ttn_data() {
  int idx, bin;
  
  for (int i=0; i<12; i++) {
    idx = ttn_data[i*4]*256 + ttn_data[i*4+1];
    bin = ttn_data[i*4+2]*256 + ttn_data[i*4+3];

    Serial.print("Bin ");
    Serial.print(idx);
    Serial.print(" = ");
    Serial.println(bin);
  }
  Serial.println("------------");
}
#endif
